##############################################################################
# "SnmpUtils" library
# A collection of shared functions related to snmp.
#
# Version : 0.10
# Author  : Y. Charton
#
# Revision history:
# 2011-06-08 v0.01   Y. Charton    First release, based from extracts of 
#                                  check_interface_table 1-3 releases
# 2011-12-30 v0.02   Y. Charton    Fusion of the get table functions
# 2012-05-02 v0.03   Y. Charton    Added transport domain selection and max-repetitions
#                                  (idea by Bjorn Frostberg), fix problem where
#                                  empty snmp query result from cache was not used
#                                  and overwritten every time, better debug
# 2012-08-29 v0.04   Y. Charton    Not empty check on result array moved inside the snmp library
#                                  OID passed to snmp functions with related info (via a hash)
# 2012-11-07 v0.05   Y. Charton    Revamped some snmp query arguments method
#                                  Added contextname (snmp v3)
# 2012-11-10 v0.06   Y. Charton    Prevent double-quoted values in unixsnmp
#                                  Force non-translated values in unixsnmp
# 2013-01-02 v0.07   Y. Charton    Specific StandardizeSNMPOutputBehaviour function
# 2013-01-19 v0.08   Y. Charton    ...
#                                  Not empty check on result array moved back to the main script
#                                  Exception handling
# 2013-10-10 v0.09   Y. Charton    Fix on StandardizeSNMPOutputBehaviour for unixsnmp mode where
#                                  the oid was cleaned of the output
# 2014-02-14 v0.10   Y. Charton    Added maxMsgSize option
#
##############################################################################

package SnmpUtils;
use strict;
use lib ('/usr/local/interfacetable_v3t/lib');
use Data::Dumper;
  $Data::Dumper::Sortkeys = 1;
use Storable;
use Exporter;
use GeneralUtils;
use Exception::Class (
    'My::Exception::Common' => {
       alias  => 'throw_error_common',
       fields => [ 'code', 'message' ] } );

##############################################################################
### I N I T ##################################################################
##############################################################################

BEGIN {
    use vars qw(@ISA @EXPORT @EXPORT_OK $VERSION $AUTHOR);
    @ISA         = qw(Exporter);
    @EXPORT      = qw(SnmpGet GetDataWithSnmp SnmpGetTable GetTableDataWithSnmp ExecuteCommand StandardizeSNMPOutputBehaviour);
    @EXPORT_OK   = qw();
    $VERSION     = 0.08;
    $AUTHOR      = "Y. Charton";
}

##############################################################################
### S U B S ##################################################################
##############################################################################

#============================================================================#
# Single / one by one OID request via perl's netsnmp
#----------------------------------------------------------------------------#
# SnmpGet
# GetDataWithSnmp
#----------------------------------------------------------------------------#

sub SnmpGet {

    ################################
    # SUB use:
    # SUB specs:
    #  * arg 0:
    #  * return:
    ################################

    my $refhStruct = shift;

    # store variables and delete them from the hash
    # this is necessary for the snmp session which takes the same
    # hash ref and does not work with arguments other than starting
    # with a dash
    my $refaOIDs            = $refhStruct->{OID}; # ref to array of OIDs
    my $GlobalCacheDir      = $refhStruct->{CacheDir};
    my $GlobalCacheTimer    = $refhStruct->{CacheTimer};
    my $OutputHashKeyIdx    = $refhStruct->{OutputHashKeyIdx};
    my $CheckEmpty          = $refhStruct->{CheckEmpty};
    
    my $stdSNMPOutBehavior  = 1;

    delete $refhStruct->{OID};
    delete $refhStruct->{CacheDir};
    delete $refhStruct->{CacheTimer};
    delete $refhStruct->{OutputHashKeyIdx};
    delete $refhStruct->{CheckEmpty};
    
    my $refoSession;         # SNMP session object
    my $sessionError;        # SNMP session error

    my $refhQuery = ();
    if (defined $refhStruct->{'-contextname'}) {
        $refhQuery->{'-contextname'} = $refhStruct->{'-contextname'};
        delete $refhStruct->{'-contextname'};
    }

    my $refhResultOutput;
    
    # example cache dir name
    my $CacheDir = "$GlobalCacheDir/SnmpGet";

    # Create the directory if not exist
    not -d $CacheDir and MyMkdir($CacheDir);

    # create snmp session object
    ($refoSession,$sessionError) = Net::SNMP->session (%$refhStruct);
    
    if (defined $refoSession) {

        my $OIDLine;    # one line of OIDs or OIDs and caching timers
                
        # OIDs come in an array (ref) - go through each
        # example:
        #    $refaOIDs = [
        #              '.1.3.6.1.2.1.2.2.1.11.1',
        #              '.1.3.6.1.2.1.2.2.1.12.1'
        #            ];
        for $OIDLine (@$refaOIDs) {

            my $refhSnmpResult=undef;
            my $CacheTimer=0;

            # OID could be .1.3.6.1.2.1.2.2.1.11.1,200
            # <OID>,<CacheTimer> for this OID only
            my ($OID,$OIDCacheTimer) = ("","");
            ($OID,$OIDCacheTimer) = split ',',$OIDLine,2;
            $refhQuery->{'-varbindlist'} = ["$OID"];
            
            if (defined $OIDCacheTimer) {
                # remove non digits
                $OIDCacheTimer =~ s/\D//g;
                if ("X$OIDCacheTimer" eq "X") { # is empty?
                    $CacheTimer = $GlobalCacheTimer;
                } else {
                    $CacheTimer = $OIDCacheTimer;
                }
            } else {
                $CacheTimer = $GlobalCacheTimer;
            }

            my $readFromCache = 0;
            if ($CacheTimer > 0) {
                if (-r "${CacheDir}/${OID}.dat") {
                    logger(3, "Cached data file found: ${CacheDir}/${OID}.dat, cacheTimer=${CacheTimer}s");
                    my @FileProperties=stat("${CacheDir}/${OID}.dat");

                    # $FileProperties[9] = LastModifyTime of file
                    # only read the cache file if it is not too old
                    if (time - $CacheTimer < $FileProperties[9]) {
                        logger(3, " valid cached data, loading");
                        $refhSnmpResult = retrieve("${CacheDir}/${OID}.dat");
                        throw_error_common(code=>3001, message=>"Unable to retrieve from ${CacheDir}/${OID}.dat: $!\n") unless defined $refhSnmpResult;
                        # leave this subroutine with cached data found
                        $readFromCache = 1;
                    } else {
                        logger(3, " expired cached data, skipping");
                    }
                }
            }

            $refhSnmpResult = {} unless (defined $refhSnmpResult);
            # snmp value not from cache - read it from the net
            if ($readFromCache == 0) {
                $refhSnmpResult = $refoSession->get_request(%$refhQuery);
                if (defined $refhSnmpResult) {
                    logger(3, "Data collected from net: OID=${OID}, value='$refhSnmpResult->{\"$OID\"}', cacheTimer=${CacheTimer}s");
                    
                    # remove non ascii chars incl. \r and \n
                    $refhSnmpResult->{"$OID"} =~ s/[\000-\037]|[\177-\377]//g;
                    # replace ";" by ","
                    $refhSnmpResult->{"$OID"} =~ s/;/,/g;
                    # replace "|" by space
                    $refhSnmpResult->{"$OID"} =~ s/\|/ /g;
                    # remove invisible chars from the end
                    $refhSnmpResult->{"$OID"} =~ s/\s+$//g;

                    # check result validity
                    if ($stdSNMPOutBehavior == 1) {
                        $refhSnmpResult->{"$OID"} = StandardizeSNMPOutputBehaviour($refhSnmpResult->{"$OID"});
                    }
                    
                    # check empty
                    if ($CheckEmpty and $refhSnmpResult->{"$OID"} eq "") {
                        $refoSession->close();
                        logger(0, "Not expected empty value for oid \"$OID\"");
                        throw_error_common(code=>1010, message=>"Not expected empty value for oid \"$OID\"");
                    }
            
                    if ($CacheTimer > 0) {
                        logger(3, "Write cache file ${CacheDir}/${OID}.dat, CacheTimer=$CacheTimer");
                        umask "0000"; # change to rw-rw-rw maybe changed later because of security
                        store ($refhSnmpResult, "${CacheDir}/${OID}.dat") or throw_error_common(code=>3000, message=>"cannot store to ${CacheDir}/${OID}.dat: $!\n");
                    }
                } else {
                    $sessionError = $refoSession->error();
                    $refoSession->close();
                    logger(0, "SNMP error: $sessionError");
                    throw_error_common(code=>1001, message=>"SNMP error: $sessionError");
                }
            }
            
            # remove the oid and keep the index
            if ($OutputHashKeyIdx) {
                for my $key (keys %$refhSnmpResult) {
                    (my $newkey = $key) =~ s/^.*\.//g;   # remove all but the index
                    $refhSnmpResult->{$newkey} = delete $refhSnmpResult->{$key};
                }
            }
            
            # fill hash with data
            foreach my $key ( keys %$refhSnmpResult ) {
                if (defined $refhResultOutput->{$key}) {
                    $refhResultOutput->{$key} .= "|".delete $refhSnmpResult->{$key};
                } else {
                    $refhResultOutput->{$key} = delete $refhSnmpResult->{$key};
                }
            }
        }
        # finally close the session
        $refoSession->close();
    } else { 
        # session establishment problem
        logger(0, "SNMP session object creation problem: ".$sessionError);
        throw_error_common(code=>1000, message=>"SNMP session object creation problem: $sessionError");
    }

    # return the complete hash with OIDs as keys or
    # undef if the SNMP session fails
    return ($refhResultOutput);
}

# ------------------------------------------------------------------------

sub GetDataWithSnmp {

    ################################
    # SUB use: Get Data with perl net-snmp module
    # SUB specs:
    #  * arg 0: host
    #  * arg 1: hash of SNMP options
    #  * arg 2: OID
    #  * arg 3: Directory for cached data
    #  * arg 4: Cache timer
    #  * return: value retrieved by snmp
    ################################

    my $refhArgs = shift;
    
    my $CacheDir         = "/tmp/.ifCache";
    my $CacheTimer       = 0;
    my $OutputHashKeyIdx = 0;
    my $CheckEmpty       = 0;
    
    $CacheDir = $refhArgs->{'cachedir'} if (exists $refhArgs->{'cachedir'});
    $CacheDir = "$CacheDir/$refhArgs->{'host'}";
    -d "$CacheDir" or MyMkdir ("$CacheDir");
    $CacheTimer       = $refhArgs->{'cachetimer'} if (exists $refhArgs->{'cachetimer'});
    $OutputHashKeyIdx = $refhArgs->{'outputhashkeyidx'} if (exists $refhArgs->{'outputhashkeyidx'});
    $CheckEmpty       = $refhArgs->{'checkempty'} if (exists $refhArgs->{'checkempty'});
    my $stdSNMPOutBehavior  = 1;
    my $refhResult;
    my $refaCommandResult;
    
    if ( not $refhArgs->{'unixsnmp'} ) {
        # gather data via perl's snmp
        my %refhSnmpQueryOptions = (
            -hostname   => "$refhArgs->{'host'}",
            -port       => "$refhArgs->{'port'}",
            -version    => "$refhArgs->{'version'}",
            -domain     => "$refhArgs->{'domain'}",
            -timeout    => "$refhArgs->{'timeout'}",
            -retries    => "$refhArgs->{'retries'}",
            -translate  => [ -timeticks => 0x0 ], # disable conversion get raw timeticks
            OID         => $refhArgs->{'oids'},
            CacheDir    => "$CacheDir",
            CacheTimer  => int rand ($CacheTimer),  # random caching
            OutputHashKeyIdx => $OutputHashKeyIdx,
            CheckEmpty  => $CheckEmpty,
        );
        defined $refhArgs->{'maxmsgsize'} and $refhSnmpQueryOptions{'-maxmsgsize'} = "$refhArgs->{'maxmsgsize'}";
        if ( $refhArgs->{'version'} eq 3 ) {
            $refhSnmpQueryOptions{'-username'}     = "$refhArgs->{'login'}";
            $refhSnmpQueryOptions{'-authpassword'} = "$refhArgs->{'passwd'}";
            $refhSnmpQueryOptions{'-authprotocol'} = "$refhArgs->{'authproto'}";
            if (defined $refhArgs->{'privpass'} and $refhArgs->{'privpass'} ne "") {
                $refhSnmpQueryOptions{'-privpassword'} = "$refhArgs->{'privpass'}";
                $refhSnmpQueryOptions{'-privprotocol'} = "$refhArgs->{'privproto'}";
            }
            defined $refhArgs->{'contextname'} and $refhSnmpQueryOptions{'-contextname'} = "$refhArgs->{'contextname'}";
        } else {
            $refhSnmpQueryOptions{'-community'} = "$refhArgs->{'community'}";
        }
        ($refhResult) = SnmpGet (\%refhSnmpQueryOptions);
    } else {
        # gather data via unix snmpget
        my $snmpQueryCommand = "snmpget";
        my $snmpQueryOptions = "-Oqnet -t $refhArgs->{'timeout'} -r $refhArgs->{'retries'} ";
        $snmpQueryOptions .= ($refhArgs->{'version'} == 2) ? "-v 2c " : "-v $refhArgs->{'version'} ";
        if ( $refhArgs->{'version'} eq 3 ) {
            if (defined $refhArgs->{'privpass'} and $refhArgs->{'privpass'} ne "") {
                $snmpQueryOptions .= "-u $refhArgs->{'login'} -l authPriv -a $refhArgs->{'authproto'} -A $refhArgs->{'passwd'} -X $refhArgs->{'privpass'} -x $refhArgs->{'privproto'} ";
            } else {
                $snmpQueryOptions .= "-u $refhArgs->{'login'} -l authNoPriv -a $refhArgs->{'authproto'} -A $refhArgs->{'passwd'} "
            }
            if (defined $refhArgs->{'contextname'} and $refhArgs->{'contextname'} ne "") {
                $snmpQueryOptions .= "-n$refhArgs->{'contextname'} ";
            }
        } else {
            $snmpQueryOptions .= "-c $refhArgs->{'community'} ";
        }
        my $snmpQueryOID = join(' ',@{$refhArgs->{'oids'}});
        ($refaCommandResult) = ExecuteCommand ({
            Command       => "$snmpQueryCommand $snmpQueryOptions $refhArgs->{'domain'}:$refhArgs->{'host'}:$refhArgs->{'port'} $snmpQueryOID",
            Retry         => 2,
            CacheDir      => $CacheDir,
            CacheFilename => "${snmpQueryCommand}_".join('_',@{$refhArgs->{'oids'}}).".dat",
            CacheTimer    => int rand ($CacheTimer),
        });
        
        # check result validity and push to final hash
        foreach my $line (@$refaCommandResult) {
            if ($stdSNMPOutBehavior == 1) {
                $line = StandardizeSNMPOutputBehaviour($line);
            }
            if ($line ne "") {
                my ($OID,$Value) = split(' ',$line,2);
                $OID   =~ s/\s+$//g;     # remove invisible chars from the end
                $OID   =~ s/^.*\.//g if $OutputHashKeyIdx;     # remove all but the index
                $Value =~ s/\s+$//g;     # remove invisible chars from the end
                # check empty
                if ($CheckEmpty and $Value eq "") {
                    logger(0, "Not expected empty value for oid \"$OID\"");
                    throw_error_common(code=>1010, message=>"Not expected empty value for oid \"$OID\"");
                }
                if (defined $refhResult->{"$OID"}) {
                    $refhResult->{"$OID"} .= "|$Value";
                } else {
                    $refhResult->{"$OID"} = "$Value";
                }
            }
        }
    }
    
    return ($refhResult);
}


#============================================================================#
# Table OID request via perl's netsnmp or unix snmpwalk/snmpbulkwalk
#----------------------------------------------------------------------------#
# SnmpGetTable
#
#----------------------------------------------------------------------------#

sub SnmpGetTable {

    ################################
    # SUB use:
    # SUB specs:
    #  * arg 0:
    #  * return:
    ################################

    my $refhStruct = shift;

    # store variables and delete them from the hash
    # this is necessary for the snmp session which takes the same
    # hash ref and does not work with arguments other than starting
    # with a dash
    my $refaOIDs            = $refhStruct->{OID}; # ref to array of OIDs
    my $GlobalCacheDir      = $refhStruct->{CacheDir};
    my $GlobalCacheTimer    = $refhStruct->{CacheTimer};
    my $OutputHashKeyIdx    = $refhStruct->{OutputHashKeyIdx};
    my $CheckEmpty          = $refhStruct->{CheckEmpty};
    
    my $stdSNMPOutBehavior  = 1;

    delete $refhStruct->{OID};
    delete $refhStruct->{CacheDir};
    delete $refhStruct->{CacheTimer};
    delete $refhStruct->{OutputHashKeyIdx};
    delete $refhStruct->{CheckEmpty};
    
    my $refoSession;    # SNMP session object
    my $sessionError;   # SNMP session error

    my $refhQuery = ();
    if (defined $refhStruct->{'-maxrepetitions'}) {
        $refhQuery->{'-maxrepetitions'} = $refhStruct->{'-maxrepetitions'};
        delete $refhStruct->{'-maxrepetitions'};
    }
    if (defined $refhStruct->{'-contextname'}) {
        $refhQuery->{'-contextname'} = $refhStruct->{'-contextname'};
        delete $refhStruct->{'-contextname'};
    }

    my $refhErrorOutput;
    $refhErrorOutput->{code} = 0;
    $refhErrorOutput->{msg} = "";
    my $refhResultOutput;

    # example cache dir name
    my $CacheDir = "$GlobalCacheDir/SnmpGetTable";

    # Create the directory if not exist
    not -d $CacheDir and MyMkdir($CacheDir);

    # create snmp session
    ($refoSession,$sessionError) = Net::SNMP->session (%$refhStruct);

    if (defined $refoSession) {

        my $OIDLine;    # one line of OIDs or OIDs and caching timers
                
        # OIDs come in an array (ref) - go through each
        # example:
        #    $refaOIDs = [
        #              '.1.3.6.1.2.1.2.2.1.11.1',
        #              '.1.3.6.1.2.1.2.2.1.12.1'
        #            ];
        for $OIDLine (@$refaOIDs) {
        
            my $refhSnmpResult=undef;
            my  $CacheTimer=0;
    
            # OID could be .1.3.6.1.2.1.2.2.1.11.1,200
            # <OID>,<CacheTimer> for this OID only
            my ($OID,$OIDCacheTimer) = ("","");
            ($OID,$OIDCacheTimer) = split ',',$OIDLine,2;
            $refhQuery->{'-baseoid'} = "$OID";
    
            if (defined $OIDCacheTimer) {
                # remove non digits
                $OIDCacheTimer =~ s/\D//g;
                if ("X$OIDCacheTimer" eq "X") { # is empty?
                    $CacheTimer = $GlobalCacheTimer;
                } else {
                    $CacheTimer = $OIDCacheTimer;
                }
            } else {
                $CacheTimer = $GlobalCacheTimer;
            }
    
            my $readFromCache = 0;
            if ($CacheTimer > 0) {
                if (-r "${CacheDir}/${OID}.dat") {
                    logger(3, "Cached data file found: ${CacheDir}/${OID}.dat, cacheTimer=${CacheTimer}s");
                    my @FileProperties=stat("${CacheDir}/${OID}.dat");
    
                    # $FileProperties[9] = LastModifyTime of file
                    # only read the cache file if it is not too old
                    if (time - $CacheTimer < $FileProperties[9]) {
                        logger(3, " valid cached data, loading");
                        $refhSnmpResult = retrieve("${CacheDir}/${OID}.dat");
                        throw_error_common(code=>3001, message=>"Unable to retrieve from ${CacheDir}/${OID}.dat: $!\n") unless defined $refhSnmpResult;
                        # leave this subroutine with cached data found
                        $readFromCache = 1;
                    } else {
                        logger(3, " expired cached data, skipping");
                    }
                }
            }
            
            $refhSnmpResult = {} unless (defined $refhSnmpResult);
            # snmp value not from cache - read it from the net
            if ($readFromCache == 0) {
                $refhSnmpResult = $refoSession->get_table(%$refhQuery);
                if (defined $refhSnmpResult) {
                    logger(3, "Data collected from net: base-OID=${OID}, cacheTimer=${CacheTimer}s");

                    foreach my $i (keys %$refhSnmpResult) {
                        logger(3, "  oid: $i, value: $refhSnmpResult->{\"$i\"}");
                        
                        # remove non ascii chars incl. \r and \n
                        $refhSnmpResult->{"$i"} =~  s/[\000-\037]|[\177-\377]//g;
                        # replace ";" by ","
                        $refhSnmpResult->{"$i"} =~ s/;/,/g;
                        # replace "|" by space
                        $refhSnmpResult->{"$i"} =~ s/\|/ /g;
                        # remove invisible chars from the end
                        $refhSnmpResult->{"$i"} =~ s/\s+$//g;

                        # check result validity
                        if ($stdSNMPOutBehavior == 1) {
                            $refhSnmpResult->{"$i"} = StandardizeSNMPOutputBehaviour($refhSnmpResult->{"$i"});
                        }
                    }
                    
                    # check empty tree
                    if ($CheckEmpty and not %$refhSnmpResult) {
                        $refoSession->close();
                        logger(0, "Not expected empty tree for base-oid \"".$refhQuery->{'-baseoid'}."\"");
                        throw_error_common(code=>1010, message=>"Not expected empty tree for base-oid \"".$refhQuery->{'-baseoid'}."\"");
                    }
                
                    # write a cache file if the cache timer > 0
                    if ($CacheTimer > 0) {
                        logger(3, "Write cache file ${CacheDir}/${OID}.dat, CacheTimer=$CacheTimer");
                        umask "0000"; # change to rw-rw-rw maybe changed later because of security
                        store ($refhSnmpResult, "${CacheDir}/${OID}.dat") or throw_error_common(code=>3000, message=>"cannot store to ${CacheDir}/${OID}.dat: $!");
                    }
                } else {
                    $sessionError = $refoSession->error();
                    if ($sessionError =~ /Requested table is empty or does not exist/i) {
                        if ($CheckEmpty) {
                            $refoSession->close();
                            logger(0, "Not expected empty tree for base-oid \"".$refhQuery->{'-baseoid'}."\"");
                            throw_error_common(code=>1010, message=>"Not expected empty tree for base-oid \"".$refhQuery->{'-baseoid'}."\"");
                        }
                    } else {
                        $refoSession->close();
                        logger(0, "SNMP error: $sessionError");
                        throw_error_common(code=>1001, message=>"SNMP error: $sessionError");
                    }
                }
            }
            
            # remove the oid and keep the index
            if ($OutputHashKeyIdx) {
                for my $key (keys %$refhSnmpResult) {
                    (my $newkey = $key) =~ s/^\.*${OID}\.//g;   # remove all but the index
                    $refhSnmpResult->{$newkey} = delete $refhSnmpResult->{$key};
                }
            }

            # fill hash with data
            #@$refhResultOutput{keys %$refhSnmpResult} = values %$refhSnmpResult;
            foreach my $key ( keys %$refhSnmpResult ) {
                if (defined $refhResultOutput->{$key}) {
                    $refhResultOutput->{$key} .= "|".delete $refhSnmpResult->{$key};
                } else {
                    $refhResultOutput->{$key} = delete $refhSnmpResult->{$key};
                }
            }
        }

        # finally close the session
        $refoSession->close();
    } else {
        # session establishment problem
        logger(0, "SNMP session object creation problem: ".$sessionError);
        throw_error_common(code=>1000, message=>"SNMP session object creation problem: $sessionError");
    }
      
    return ($refhResultOutput);
}

#----------------------------------------------------------------------------#

sub GetTableDataWithSnmp {

    ################################
    # SUB use: Get multiple Data with perl net-snmp module
    #          or unix snmpwalk/snmpbulkwalk
    # SUB specs:
    #  * arg 0: host
    #  * arg 1: hash of SNMP options
    #  * arg 2: hash of info on the target OID
    #  * arg 3: Directory for cached data
    #  * arg 4: Cache timer
    #  * return: hash of values retrieved by snmp
    ################################

    my $refhArgs = shift;
    
    my $CacheDir         = "/tmp/.ifCache";
    my $CacheTimer       = 0;
    my $OutputHashKeyIdx = 0;
    my $CheckEmpty       = 0;
    
    $CacheDir = $refhArgs->{'cachedir'} if (exists $refhArgs->{'cachedir'});
    $CacheDir = "$CacheDir/$refhArgs->{'host'}";
    -d "$CacheDir" or MyMkdir ("$CacheDir");
    $CacheTimer       = $refhArgs->{'cachetimer'} if (exists $refhArgs->{'cachetimer'});
    $OutputHashKeyIdx = $refhArgs->{'outputhashkeyidx'} if (exists $refhArgs->{'outputhashkeyidx'});
    $CheckEmpty       = $refhArgs->{'checkempty'} if (exists $refhArgs->{'checkempty'});
    my $stdSNMPOutBehavior  = 1;
    my $refhResult;
    my $refaCommandResult;
    
    if ( not $refhArgs->{'unixsnmp'} ) {
        # gather data via perl's snmp
        my %refhSnmpQueryOptions = (
            -hostname   => "$refhArgs->{'host'}",
            -port       => "$refhArgs->{'port'}",
            -version    => "$refhArgs->{'version'}",
            -domain     => "$refhArgs->{'domain'}",
            -timeout    => "$refhArgs->{'timeout'}",
            -retries    => "$refhArgs->{'retries'}",
            OID         => $refhArgs->{'oids'},
            CacheDir    => "$CacheDir",
            CacheTimer  => int rand ($CacheTimer),  # random caching;
            OutputHashKeyIdx => $OutputHashKeyIdx,
            CheckEmpty  => $CheckEmpty,
        );
        defined $refhArgs->{'maxmsgsize'} and $refhSnmpQueryOptions{'-maxmsgsize'} = "$refhArgs->{'maxmsgsize'}";
        if ( $refhArgs->{'version'} eq 3 ) {
            $refhSnmpQueryOptions{'-username'}     = "$refhArgs->{'login'}";
            $refhSnmpQueryOptions{'-authpassword'} = "$refhArgs->{'passwd'}";
            $refhSnmpQueryOptions{'-authprotocol'} = "$refhArgs->{'authproto'}";
            if (defined $refhArgs->{'privpass'} and $refhArgs->{'privpass'} ne "") {
                $refhSnmpQueryOptions{'-privpassword'} = "$refhArgs->{'privpass'}";
                $refhSnmpQueryOptions{'-privprotocol'} = "$refhArgs->{'privproto'}";
            }
            defined $refhArgs->{'max-repetitions'} and $refhSnmpQueryOptions{'-maxrepetitions'} = "$refhArgs->{'max-repetitions'}";
            defined $refhArgs->{'contextname'} and $refhSnmpQueryOptions{'-contextname'} = "$refhArgs->{'contextname'}";
        } elsif ( $refhArgs->{'version'} eq 2 ) {
            $refhSnmpQueryOptions{'-community'} = "$refhArgs->{'community'}";
            defined $refhArgs->{'max-repetitions'} and $refhSnmpQueryOptions{'-maxrepetitions'} = "$refhArgs->{'max-repetitions'}";
        } else {
            $refhSnmpQueryOptions{'-community'} = "$refhArgs->{'community'}";
        }
        ($refhResult) = SnmpGetTable (\%refhSnmpQueryOptions);
    } else {
        # gather data via unix snmpwalk/snmpbulkwalk
        my $snmpQueryCommand = "snmpbulkwalk";
        my $snmpQueryOptions = "-Oqnet -t $refhArgs->{'timeout'} -r $refhArgs->{'retries'} ";
        $snmpQueryOptions .= ($refhArgs->{'version'} == 2) ? "-v 2c " : "-v $refhArgs->{'version'} ";
        if ( $refhArgs->{'version'} eq 3 ) {
            if (defined $refhArgs->{'privpass'} and $refhArgs->{'privpass'} ne "") {
                $snmpQueryOptions .= "-u $refhArgs->{'login'} -l authPriv -a $refhArgs->{'authproto'} -A $refhArgs->{'passwd'} -X $refhArgs->{'privpass'} -x $refhArgs->{'privproto'} ";
            } else {
                $snmpQueryOptions .= "-u $refhArgs->{'login'} -l authNoPriv -a $refhArgs->{'authproto'} -A $refhArgs->{'passwd'} "
            }
            defined $refhArgs->{'max-repetitions'} and $snmpQueryOptions .= "-Cr$refhArgs->{'max-repetitions'} ";
            if (defined $refhArgs->{'contextname'} and $refhArgs->{'contextname'} ne "") {
                $snmpQueryOptions .= "-n$refhArgs->{'contextname'} ";
            }
        } elsif ( $refhArgs->{'version'} eq 2 ) {
            $snmpQueryOptions .= "-c $refhArgs->{'community'} ";
            defined $refhArgs->{'max-repetitions'} and $snmpQueryOptions .= "-Cr$refhArgs->{'max-repetitions'} ";
        } else {
            $snmpQueryCommand = "snmpwalk";
            $snmpQueryOptions .= "-Cc -c $refhArgs->{'community'} ";
        }
        foreach my $singleoid (@{$refhArgs->{'oids'}}) {
            ($refaCommandResult) = ExecuteCommand ({
                Command       => "$snmpQueryCommand $snmpQueryOptions $refhArgs->{'domain'}:$refhArgs->{'host'}:$refhArgs->{'port'} $singleoid",
                Retry         => 2,
                CacheDir      => $CacheDir,
                CacheFilename => "${snmpQueryCommand}_${singleoid}.dat",
                CacheTimer    => int rand ($CacheTimer),
            });
            
            # check empty tree
            if ($CheckEmpty and not $refaCommandResult) {
                logger(0, "Not expected empty tree for base-oid \"$singleoid\"");
                throw_error_common(code=>1010, message=>"Not expected empty tree for base-oid \"$singleoid\"");
            }
            
            # check result validity and push to final hash
            foreach my $line (@$refaCommandResult) {
                if ($stdSNMPOutBehavior == 1) {
                    $line = StandardizeSNMPOutputBehaviour($line);
                }
                if ($line ne "") {
                    my ($OID,$Value) = split(' ',$line,2);
                    $OID =~ s/\s+$//g;                                      # remove invisible chars from the end
                    $OID =~ s/^\.*$singleoid\.//g if $OutputHashKeyIdx;     # remove all but the index
                    $Value =~ s/\s+$//g;                                    # remove invisible chars from the end
                    if (defined $refhResult->{"$OID"}) {
                        $refhResult->{"$OID"} .= "|$Value";
                    } else {
                        $refhResult->{"$OID"} = "$Value";
                    }
                }
            }
        }
    }

    return ($refhResult);
}


#============================================================================#
# Other functions
#----------------------------------------------------------------------------#
# ExecuteCommand
# StandardizeSNMPOutputBehaviour
#----------------------------------------------------------------------------#

# ------------------------------------------------------------------------
# ExecuteCommand Routine. Enhanced with our cache algorith...

sub ExecuteCommand {
    my $refhStruct      = shift;
    my $Command         = $refhStruct->{Command};
    my $GlobalCacheDir  = $refhStruct->{CacheDir};
    my $CacheFilename   = $refhStruct->{CacheFilename} if (defined $refhStruct->{CacheFilename});

    my $refaLines;      # Pointer to Array of strings (output)
    my $CacheFile;      # Filename storing cached data
    my $Now = time();   # current time in seconds since epoch
    my $CacheDir = "$GlobalCacheDir/ExecuteCommand/"; # cache dir

    my $refhResultOutput;
    
    # Create Cachedir if not exist
    not -d $CacheDir and MyMkdir($CacheDir);

    # If caching for this command is enabled
    if ($refhStruct->{CacheTimer} > 0) {
        if (defined $CacheFilename) {
            $CacheFile = $CacheDir . $CacheFilename;
        } else {
            $CacheFile = $CacheDir . normalize ("$Command") . ".dat";
        }
        if (-r "$CacheFile") {
            logger(3, "Cached data file found: $CacheFile, cacheTimer=$refhStruct->{CacheTimer}s");
            my @FileProperties=stat($CacheFile);

            # $FileProperties[9] = LastModifyTime of file
            # only read the cache file if it is not too old
            if ($Now-$refhStruct->{CacheTimer} < $FileProperties[9]) {
                logger(3, " valid cached data, loading");
                $refaLines = retrieve($CacheFile);
                throw_error_common(code=>3001, message=>"Unable to retrieve from ${CacheFile}: $!\n") unless defined $refaLines;
                foreach (@$refaLines) {
                    $_ =~ s/\n$//g;
                    logger(3, "   data: $_");
                }
                # leave this subroutine with cached data found
                return ($refaLines);
            } else {
                logger(3, " expired cached data, skipping");
            }
        }
    }

    # execute the unix command
    open(UNIX,"$Command 2>&1|") or throw_error_common(code=>$?>>8, message=>"Failed to execute \"$Command\", $!");
        while (<UNIX>) {
            $_ =~ s/^\.*([0-9.]*) "?(.*)/$1 $2/g;
            $_ =~ s/"$//g;
            $_ =~ s/\|/ /g; # remove pipes which could produce problems (ex: with nagios performance data separator)
            $_ =~ s/\n$//g; # remove ending return carriers
            push @$refaLines,$_;
        }
    close(UNIX);
    my $ErrorCode = $? >> 8; # calculate the exit code
    logger(3, "Executed \"$Command\" and got exit code \"$ErrorCode\"");

    # stop in case of error
    if ($ErrorCode) {
        my $ErrorMessage =  join(', ',@$refaLines) if (defined $refaLines->[0]);
        $ErrorMessage =~ s/\s+$//g;
        throw_error_common(code=>$ErrorCode, message=>"SNMP error: $ErrorMessage");
    }

    # write a cache file if the cache timer > 0
    if ($refhStruct->{CacheTimer} > 0) {
        logger(3, "Write cache file $CacheFile, CacheTimer=$refhStruct->{CacheTimer}");
        umask "0000"; # change to rw-rw-rw maybe changed later because of security
        store ($refaLines, "$CacheFile") or throw_error_common(code=>3000, message=>"cannot store to $CacheFile: $!");
    }
    return ($refaLines);
}

sub StandardizeSNMPOutputBehaviour {
    my $snmpResult = shift;
    if ($snmpResult =~ /noSuchObject/) {
        # Replacing noSuchObject by an empty string.
        # Indeed, the different snmp versions handle requests for missing oids
        # differently. v1 always returns an empty string. v2c/v3 return "noSuchObject"
        # string, and -translate noSuchObject -> '' correct that. BUt we do the change
        # to empty string after to be able to inform the user. Also, snmpwalk and other
        # unix programs don't have such a translate option.
        # Ex: snmpwalk
        # [icinga@server libexec]$ snmpwalk -Oqn -v 1 -t 15 -c community xxxxxxxxxxx:161 .1.3.6.1.4.1.9.5.1.2.16.0
        # [icinga@server libexec]$
        # [icinga@server libexec]$ snmpwalk -Oqn -v 2c -t 15 -c community xxxxxxxxxxx:161 .1.3.6.1.4.1.9.5.1.2.16.0
        # .1.3.6.1.4.1.9.5.1.2.16.0 No Such Object available on this agent at this OID
        # [icinga@server libexec]$
        $snmpResult =~ s/noSuchObject.*$//;
        logger(3, "Data validity check: no such objet in the mib. SNMP behavior standardization enabled, replacing by ''");
    }
    elsif ($snmpResult =~ /noSuchInstance/) {
        # Replacing noSuchInstance by an empty string.
        # Indeed, the different snmp versions handle requests for missing oids
        # differently. v1 always returns an empty string. v2c/v3 return "noSuchInstance"
        $snmpResult =~ s/noSuchInstance.*$//;
        logger(3, "Data validity check: no such objet in the mib. SNMP behavior standardization enabled, replacing by ''");
    }
    elsif ($snmpResult =~ /Received noSuchName/) {
        # Replacing noSuchName by an empty string.
        # Indeed, the different snmp versions handle requests for missing oids
        # differently. v1 always returns an empty string ?????. v2c/v3 return "noSuchName"
        $snmpResult =~ s/Received noSuchName.*$//;
        logger(3, "Data validity check: no such objet in the mib. SNMP behavior standardization enabled, replacing by ''");
    }
    elsif ($snmpResult =~ /No Such Object available on this agent at this OID/) {
        # Replacing "No Such Object available on this agent at this OID" by an empty string.
        # Indeed, the different snmp versions handle requests for missing oids
        # differently. While v1 returns an empty string, v2c/v3 return "noSuchObject"
        # string
        # Ex: snmpwalk
        # [icinga@server libexec]$ snmpwalk -Oqn -v 1 -t 15 -c community xxxxxxxxxxx:161 .1.3.6.1.4.1.9.5.1.2.16.0
        # [icinga@server libexec]$
        # [icinga@server libexec]$ snmpwalk -Oqn -v 2c -t 15 -c community xxxxxxxxxxx:161 .1.3.6.1.4.1.9.5.1.2.16.0
        # .1.3.6.1.4.1.9.5.1.2.16.0 No Such Object available on this agent at this OID
        # [icinga@server libexec]$
        #
        #30/08/2012: commenting return to force empty check 
        #logger(2, "Data validity check: $line, no such objet in the mib. Strict mode disabled, returning an empty array");
        #return $refaLines;
        $snmpResult =~ s/No Such Object.*$//;
        logger(3, "Data validity check: no such objet in the mib. SNMP behavior standardization enabled, replacing by ''");
    }
    elsif ($snmpResult =~ /No Such Instance currently exists at this OID/) {
        # Replacing "No Such Instance currently exists at this OID" by an empty string.
        # Indeed, the different snmp versions handle requests for missing oids
        # differently. v1 returns an "Error in packet" message, v2c/v3 return "noSuchInstance"
        # string
        # Ex: snmpwalk
        # [root@snoopy ~]# snmpget -Oqn -v 1 -t 15 -c public localhost:161 sysDescr
        # Error in packet
        # Reason: (noSuchName) There is no such variable name in this MIB.
        # Failed object: .1.3.6.1.2.1.1.1
        # [icinga@server libexec]$
        # [root@snoopy ~]# snmpget -Oqn -v 2c -t 15 -c public localhost:161 sysDescr
        # .1.3.6.1.2.1.1.1 No Such Instance currently exists at this OID
        # [icinga@server libexec]$
        #
        #30/08/2012: commenting return to force empty check 
        #logger(2, "Data validity check: $line, no such objet in the mib. Strict mode disabled, returning an empty array");
        #return $refaLines;
        $snmpResult =~ s/No Such Instance.*$//;
        logger(3, "Data validity check: no such objet in the mib. SNMP behavior standardization enabled, replacing by ''");
    }
    return $snmpResult;
}


1;

__END__

# vi: set ts=4 sw=4 expandtab :
