##############################################################################
# "GeneralUtils" library
# A collection of shared functions for general usage.
#
# Version 0.03
# Written by Y. Charton, parts based on some extracts from Nconf and NagiosBp 
# libraries
#
# Revision history:
# 2011-06-08 v0.01   Y. Charton    First release
# 2012-08-26 v0.02   Y. Charton    Added hexToString and stringToHex
#                                  Some conversion functions
# 2013-02-07 v0.03   Y. Charton    Added log to file, changed volume formating
# 2013-10-15 v0.04   Y. Charton    Denormalize func fix when text empty
#
##############################################################################

package GeneralUtils;
use strict;
use lib ('/usr/local/interfacetable_v3t/lib');
use Exporter;

use vars qw($LOGLEVEL $LOGDEST);
$LOGLEVEL = 0;
$LOGDEST = "";

##############################################################################
### I N I T ##################################################################
##############################################################################

BEGIN {
    use vars qw(@ISA @EXPORT @EXPORT_OK $VERSION $AUTHOR);
    @ISA         = qw(Exporter);
    @EXPORT      = qw(setLoglevel setLogdest logger printArray printHash fixedLen cutOffSpaces getHostnameFromUrl getProtocolFromUrl MyMkdir normalize denormalize trim stringToHex hexToString format_volume_decimal format_volume_binary);
    @EXPORT_OK   = qw();
    $VERSION     = 0.3;
    $AUTHOR      = "Y. Charton";
}

##############################################################################
### S U B S ##################################################################
##############################################################################

#----------------------------------------------------------------------------#
# List:
# setLoglevel
# setLogdest
# logger
# printArray
# printHash
# fixedLen
# cutOffSpaces
# getHostnameFromUrl
# getProtocolFromUrl
# MyMkdir
# normalize
# denormalize
# trim
# stringToHex
# hexToString
# bytes2bits
# unit2bytes
# unit2bits
# format_volume_decimal
# format_volume_binary
#----------------------------------------------------------------------------#


sub setLoglevel {

    ################################
    # SUB use: Set the overall loglevel
    # SUB specs: ###################
    #  * arg 0: The loglevel to set (1-5)
    ################################

    my $loglevel = shift;
    if($loglevel >= 1 && $loglevel <= 5){
        $LOGLEVEL = $loglevel;
    }
}

#----------------------------------------------------------------------------#

sub setLogdest {

    ################################
    # SUB use: Set destination log file
    # SUB specs: ###################
    #  * arg 0: the destination log file
    ################################

    my $logdest = shift;
    $LOGDEST = $logdest;
}

#----------------------------------------------------------------------------#

sub logger {

    ################################
    # SUB use: Log and display messages based on loglevel
    # SUB specs:
    #  * arg 0: The loglevel of the message (0-5)
    #  * arg 1: The message to be logged
    #  * return: nothing (all is printed to STDOUT)
    ################################
    # loglevel - message category
    #        0 - criticals
    #        1 - information
    #        1 - warnings
    #        2 - calculations
    #        2 - html output
    #        2 - performance data output
    #        3 - snmp queries/results
    #        3 - executecommand
    #        3 - commandline
    #        3 - config file
    #        3 - options
    #        3 - history file content
    #        3 - current status
    #        5 - hash/array dumps
    ################################

    my $loglevel = 0;
    my $msg      = undef;

    $loglevel    = shift;
    $msg         = shift;

    unless($loglevel >= 0 and $loglevel <= 5){die "You passed a message with an invalid loglevel to logger().\n"}
    unless($LOGLEVEL >= 0 and $LOGLEVEL <= 5){die "The main loglevel is not set properly (must be 0-5).\n"}

    # define labels for different loglevels
    my %level_labels = ( 0 => "ERROR", 1 => "INFO", 2 => "DEBUG", 3 => "TRACE1", 4 => "TRACE2", 5 => "DUMP" );

    # fetch the name of the function which called logger()
    my $caller = (caller(1))[3];
    if($caller){
        $caller="|".$caller."| ";
    }else{
        $caller="|main| ";
    }

    # printing the message depending on the debug level
    if($loglevel <= $LOGLEVEL and $LOGLEVEL > 0){
        my $space = "";
        if(length($level_labels{$loglevel}) == 4){$space=" "}
        # show the calling function, if loglevel is higher than 2
        if($LOGLEVEL >= 2){
            print "[$level_labels{$loglevel}]$space $caller $msg\n";
        }else{
            print "[$level_labels{$loglevel}]$space $msg\n";

        }
        if ($LOGDEST) {
            umask "0222";
            open (FILE,">>$LOGDEST") or die "cannot open $LOGDEST $!";
            flock (FILE, 2) or die "cannot flock $LOGDEST ($!)"; # get exclusive lock;
            if($LOGLEVEL >= 2){
                print FILE "[$level_labels{$loglevel}]$space $caller $msg\n";
            }else{
                print FILE "[$level_labels{$loglevel}]$space $msg\n";
            }
            close(FILE);
        }
    }
}

#----------------------------------------------------------------------------#

sub printArray {
    my $array = shift;
    my $prefix = shift;
    my $output = "";
    my ($i, $len);

    if (defined $array) {
        $len=length(@$array - 1);
        #$output = $output . "len: $len";
        for ($i=0; $i<@$array; $i++)
        {
            $i = sprintf("%0${len}d", $i);
            if ($output ne "") {
                $output = $output . '\n';
            }
            $output = $output . "${prefix}[$i]: $array->[$i]";
        }
    }
    return $output;
}

#----------------------------------------------------------------------------#

sub printHash {
    my $hash = shift;
    my $prefix = shift || "";
    my $output = "";
    my ($key);
    my $maxlen = 0;

    if(defined $hash) {
        foreach $key (keys %$hash)
        {
            if (length($key) > $maxlen)
            {
                $maxlen = length($key);
            }
        }
        #$output = $output . "max: $maxlen\n";
        foreach $key (keys %$hash)
        {
            if ($output ne "") {
                $output = $output . '\n';
            }
            $output = $output . "${prefix}" . fixedLen("[$key]", $maxlen+2, "left") . " => $hash->{$key}";
        }
    }
    return $output;
}

# ------------------------------------------------------------------------

sub fixedLen {

    ################################
    # SUB use: GeneralUtils::fixedLen($string [, $len [, "left"|"right" [, $fillchar]]])
    #          brings a given string to a fixed length and returns the string afterwards
    #          no matter if it is shorter or longer before
    # SUB specs:
    #  * arg 0: the string
    #  * arg 1: the desired length (integer), defaults to 10 if omitted
    #  * arg 2: "left" or "right": tells on which side blanks are appended or characters are cut off
    #  * arg 3: fillcharacter: 1 character, which should be used to fill up short strings
    #           defaults to blank " "
    #  * return: the resulting string
    ################################

    my $string = shift;
    my $len = shift || 10;
    my $side = shift || "right";
    my $fillchar = shift || " ";
    my $fillstring;

    if (length($string) > $len)
    {
        if ($side eq "left")
        {
            $string = substr($string, $len*(-1));
        }
        else
        {
            $string = substr($string, 0, $len);
        }
    }
    if (length($string) < $len)
    {
        $fillchar = substr($fillchar, 0, 1);
        $fillstring = $fillchar x ($len-length($string));
        if ($side eq "left")
        {
            $string = $fillstring . $string;
        }
        else
        {
            $string .= $fillstring;
        }
    }

    return $string;
}

# ------------------------------------------------------------------------

sub cutOffSpaces {

    ################################
    # SUB use: cuts of leading and trailing whitespace characters of a given string
    # SUB specs:
    #  * arg 0: the string
    #  * return: the resulting string
    ################################

    my $string = shift;
    $string =~ s/^\s*//;
    $string =~ s/\s*$//;
    # does the same as the two lines above, but takes twice as long
    #$string =~ s/^\s*(.*?)\s*$/$1/;
    return ($string);
}

# ------------------------------------------------------------------------

sub getHostnameFromUrl {

    ################################
    # SUB use: from a given URL, we extract the hostname
    #          give "s" as second parameter to get the short hostname (everything before the first dot)
    #          give "l" or leave empty, to get the full qualified hostname, if it is in the URL as full qualified name
    # SUB specs:
    #  * arg 0: the URL
    #  * arg 1: the return modifier
    #  * return: the hostname as string
    ################################

    my $url = shift;
    my $switch = shift;

    if ($switch eq "s")
    {
        # if an IP is used instead a hostname there is no sense in cutting after the first dot
        if ($url =~ m/^(.+:\/\/)?(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})/)
        {
            if (defined($2))
            {
                return($2);
            }
            else
            {
                return($1);
            }
        }

        $url =~ m/^(.+:\/\/)?([^.:\/]+)/;
        if (defined($2))
        {
            return($2);
        }
        else
        {
            return($1);
        }
    }
    else
    {
        $url =~ m/^(.+:\/\/)?([^:\/]+)/;
        if (defined($2))
        {
            return($2);
        }
        else
        {
            return($1);
        }
    }
    return(undef);
}

# ------------------------------------------------------------------------

sub getProtocolFromUrl {

    ################################
    # SUB use: From a given URL, we extract the protocol
    # SUB specs:
    #  * arg 0: the URL
    #  * arg 1: The message to be logged
    #  * return: the protocol as string
    ################################

    my $url = shift;
    $url =~ m/^(.+):\/\//;
    if (defined($1))
    {
        return($1);
    }
    else
    {
        return("http");
    }
}

# ------------------------------------------------------------------------
# dir creator extracted from our FileOperations library

sub MyMkdir {

    my  $Directory = shift;

    $Directory  =~  s/\s//g;    # remove invisible characters
    $Directory  =~  s/\\/\//g;  # replace \ with /
    $Directory  =~  s/\/+/\//g; # replace duplicate or more slashes with one /

    # split directory into pieces
    my @Dirs    =   split /\//,$Directory;

    # growing directory structure
    my $Snake = "";

    # walk through each directory and create it
    for my $Dir (@Dirs) {
        $Snake .= $Dir.'/';
        if (not -d "$Snake") {
            umask "0000";
            my $Success = mkdir $Snake;
            if ("$Success" ne "1") {
                undef $Snake;
                last;
            }
        }
    }

    # remove last / and return the directory back to the caller
    if ($Snake) {
            if ("$Snake" =~ /\/$/) {
            chop ($Snake);
        }
        }
    return $Snake;
}

# ------------------------------------------------------------------------
# normalize and denormalize subroutines
# - used to get rid of special characters

sub normalize {
    my $Text=shift;
    $Text=~s/Q/Q51/g;                                        # Q normalisieren
    $Text=~s/[\000-\017]/sprintf ("Q0%X",ord($&))/ge;        # einstellige HEX Zahlen mit 0 davor konvertieren
    $Text=~s/[\W_]/sprintf ("Q%X",ord($&))/ge;               # alle anderen Zeichen durch Q + HEX code ersetzen
    return $Text;                                            # fertig
}
sub denormalize {
    my $Text=shift;
    $Text=~s/Q../pack "H2",substr($&,1,3)/ge;
    if (not defined $Text) {
        $Text = "";
    }
    return $Text;
}

# ------------------------------------------------------------------------
# trim function to remove whitespace from the start and end of the string

sub trim {
    my @out = @_;
    for (@out) {
        s/^\s+//;
        s/\s+$//;
    }
    return wantarray ? @out : $out[0];
}

# ------------------------------------------------------------------------
# string from/to hex conversion

sub stringToHex {
    my $string=shift;
    my $hstring="";
    logger(5, "To convert: $string");  
    # Convert to hex
    $hstring = unpack ("H*",$string);
    logger(5, "Converted to: $hstring");  
    return $hstring;                                            
}
sub hexToString {
    my $hstring=shift;
    my $string="";
    logger(5, "To convert: $hstring");  
    if ($hstring =~ m/0x[0-9a-f]+/i) {
        # Convert to string
        $string = pack (qq{H*},qq{$hstring});
        logger(5, "Converted to: $string");  
    } else {
        logger(5, "Not a hexadecimal, no conversion");
        $string="$hstring";
    }
    return $string;                                            
}

# ------------------------------------------------------------------------
# bytes and bits conversions

sub bytes2bits {
 return unit2bytes(@_)*8;
}

#Converts an input value to value in bytes
sub unit2bytes {
 my ( $value, $unit ) = @_;
 #given ($unit) {
 # when ('T') { return $value*1099511627776; }
 # when ('G') { return $value*1073741824; }
 # when ('M') { return $value*1048576; }
 # when ('K') { return $value*1024; }
 # default { return $value }
 #};
 if ($unit eq 'T') {
   return $value*1099511627776;
 } 
 elsif ($unit eq 'G') {
   return $value*1073741824;
 } 
 elsif ($unit eq 'M') {
   return $value*1048576;
 }
 elsif ($unit eq 'K') {
   return $value*1024;
 }
 else { 
   return $value;
 }
}

sub unit2bits {
 my ( $value, $unit ) = @_;
 #given ($unit) {
 # when ('t') { return $value*1000000000000; }
 # when ('g') { return $value*1000000000; }
 # when ('m') { return $value*1000000; }
 # when ('k') { return $value*1000; }
 # default { return $value }
 #};
 if ($unit eq 't') {
   return $value*1_000_000_000_000;
 }
 elsif ($unit eq 'g') {
   return $value*1_000_000_000;
 }
 elsif ( $unit eq 'm' ) {
   return $value*1_000_000;
 }
 elsif ( $unit eq 'k' ) {
   return $value*1000
 } 
 else { return $value }

};

sub format_volume_decimal {
 my $volume=shift;
 my $suffix=shift;
 my $prefix="";
 
 if ( $volume>=1000000000000000000 ) {
  $volume=$volume/1000000000000000000;
  $prefix="E";
 } 
 elsif ( $volume>=1000000000000000 ) {
  $volume=$volume/1000000000000000;
  $prefix="P";
 } 
 elsif ( $volume>=1000000000000 ) {
  $volume=$volume/1000000000000;
  $prefix="T";
 } 
 elsif ( $volume>=1000000000 ) {
  $volume=$volume/1000000000;
  $prefix="G";
 }
 elsif ( $volume>=1000000 ) {
  $volume=$volume/1000000;
  $prefix="M";
 }
 elsif ( $volume>=1000 ) {
  $volume=$volume/1000;
  $prefix="K";
 }
 $volume=sprintf("%.2f",$volume);
 return $volume." ".$prefix.$suffix;
}

sub format_volume_binary {
 my $volume=shift;
 my $suffix=shift;
 my $prefix="";
 
 if ( $volume>=1152921504606846976 ) {
  $volume=$volume/1152921504606846976;
  $prefix="E";
 }
 elsif ( $volume>=1125899906842624 ) {
  $volume=$volume/1125899906842624;
  $prefix="P";
 }
 elsif ( $volume>=1099511627776 ) {
  $volume=$volume/1099511627776;
  $prefix="T";
 }
 elsif ( $volume>=1073741824 ) {
  $volume=$volume/1073741824;
  $prefix="G";
 }
 elsif ( $volume>=1048576 ) {
  $volume=$volume/1048576;
  $prefix="M";
 }
 elsif ( $volume>=1024 ) {
  $volume=$volume/1024;
  $prefix="K";
 }
 $volume=sprintf("%.2f",$volume);
 return $volume." ".$prefix.$suffix;
}

1;

__END__

=head1 NAME

 GeneralUtils - Some functions quite often used

=head1 SYNOPSIS

    use GeneralUtils;

    printArray(\@a);
    printArray(\@a, "DEBUG: ");

    printHash(\%h);
    printHash(\%h, "    ");  # e. g. if you want to print intended

    print fixedLen("aVeryLongStringWithoutAnySpace", 20, "right") . "\n";
    # prints out: aVeryLongStringWitho

    $c = "short";
    $c = fixedLen($c, 20, undef, ".");
    # $c is now: short...............

    $string = cutOffSpaces("foo   ");
    # $string = "foo"

    $hostname = getHostnameFromUrl("http://www.example.com:80/foo/");
    # $hostname = "www.example.com";
    $hostname = getHostnameFromUrl("http://myworkstation.example.com", "s");
    # $hostname = "myworkstation";

    print getProtocolFromUrl("https://www.example.com/test/") ."\n";
    # https
    $p = getProtocolFromUrl("www.example.com");
    # $p = "http";

=head1 DESCRIPTION

=head2 GeneralUtils::printArray

    GeneralUtils::printArray(\@array [, $prefix])

    prints out the content of an array in a structured way

    parameter 1: reference to an array
    parameter 2: prefix for every line of output
    returns:     nothing of value

=head2 GeneralUtils::printHash

    GeneralUtils::printHash(\%hash [, $prefix])

    prints out the content of a hash in a structured way

    parameter 1: reference to a hash
    parameter 2: prefix for every line of output
    returns:     nothing of value

=cut

# vi: set ts=4 sw=4 expandtab :
